<?php
/**
 * Nextcloud setup wizard
 *
 * @author Frank Karlitschek
 * @copyright 2012 Frank Karlitschek frank@owncloud.org
 * @author Lukas Reschke
 * @copyright 2013-2015 Lukas Reschke lukas@owncloud.com
 * @copyright 2016 Lukas Reschke lukas@statuscode.ch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU AFFERO GENERAL PUBLIC LICENSE
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU AFFERO GENERAL PUBLIC LICENSE for more details.
 *
 * You should have received a copy of the GNU Affero General Public
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/**
 * Please copy this file into your webserver root and open it with a browser. The setup wizard checks the dependency, downloads the newest Nextcloud version, unpacks it and redirects to the Nextcloud first run wizard.
 */

// Nextcloud version
define('NC_VERSION', '16.0.3');

// init
ob_start();
error_reporting(E_ERROR | E_WARNING | E_PARSE | E_NOTICE);
ini_set('display_errors', 1);
@set_time_limit(0);

/**
 * Setup class with a few helper functions
 */
class Setup {

	private static $requirements = array(
		array(
			'classes' => array(
				'ZipArchive' => 'zip',
				'DOMDocument' => 'dom',
				'XMLWriter' => 'XMLWriter'
			),
			'functions' => array(
				'xml_parser_create' => 'libxml',
				'mb_detect_encoding' => 'mb multibyte',
				'ctype_digit' => 'ctype',
				'json_encode' => 'JSON',
				'gd_info' => 'GD',
				'gzencode' => 'zlib',
				'iconv' => 'iconv',
				'simplexml_load_string' => 'SimpleXML',
				'hash' => 'HASH Message Digest Framework',
				'curl_init' => 'curl',
			),
			'defined' => array(
				'PDO::ATTR_DRIVER_NAME' => 'PDO'
			),
		)
	);


	/**
	* Checks if all the Nextcloud dependencies are installed
	* @return string with error messages
	*/
	static public function checkDependencies() {
		$error = '';
		$missingDependencies = array();

		// do we have PHP 7.1.0 or newer?
		if(version_compare(PHP_VERSION, '7.1.0', '<')) {
			$error.='PHP 7.1.0 is required. Please ask your server administrator to update PHP to version 7.1.0 or higher.<br/>';
		}

		// running oC on windows is unsupported since 8.1
		if(substr(PHP_OS, 0, 3) === "WIN") {
			$error.='Nextcloud Server does not support Microsoft Windows.<br/>';
		}

		foreach (self::$requirements[0]['classes'] as $class => $module) {
			if (!class_exists($class)) {
				$missingDependencies[] = array($module);
			}
		}
		foreach (self::$requirements[0]['functions'] as $function => $module) {
			if (!function_exists($function)) {
				$missingDependencies[] = array($module);
			}
		}
		foreach (self::$requirements[0]['defined'] as $defined => $module) {
			if (!defined($defined)) {
				$missingDependencies[] = array($module);
			}
		}

		if(!empty($missingDependencies)) {
			$error .= 'The following PHP modules are required to use Nextcloud:<br/>';
		}
		foreach($missingDependencies as $missingDependency) {
			$error .= '<li>'.$missingDependency[0].'</li>';
		}
		if(!empty($missingDependencies)) {
			$error .= '</ul><p style="text-align:center">Please contact your server administrator to install the missing modules.</p>';
		}

		// do we have write permission?
		if(!is_writable('.')) {
			$error.='Can\'t write to the current directory. Please fix this by giving the webserver user write access to the directory.<br/>';
		}

		return($error);
	}


	/**
	* Check the cURL version
	* @return bool status of CURLOPT_CERTINFO implementation
	*/
	static public function isCertInfoAvailable() {
		$curlDetails =  curl_version();
		return version_compare($curlDetails['version'], '7.19.1') != -1;
	}

	/**
	* Performs the Nextcloud install.
	* @return string with error messages
	*/
	static public function install() {
		$error = '';
		$directory = trim($_GET['directory']);

		// Test if folder already exists
		if(file_exists('./'.$directory.'/status.php')) {
			return 'The selected folder seems to already contain a Nextcloud installation. - You cannot use this script to update existing installations.';
		}

		// downloading latest release
		if (!file_exists('nc.zip')) {
			$error .= Setup::getFile('https://download.nextcloud.com/server/releases/nextcloud-'.NC_VERSION.'.zip','nc.zip');
		}

		// unpacking into nextcloud folder
		$zip = new ZipArchive;
		$res = $zip->open('nc.zip');
		if ($res==true) {
			// Extract it to the tmp dir
			$nextcloud_tmp_dir = 'tmp-nextcloud'.time();
			$zip->extractTo($nextcloud_tmp_dir);
			$zip->close();

			// Move it to the folder
			if ($_GET['directory'] === '.') {
				foreach (array_diff(scandir($nextcloud_tmp_dir.'/nextcloud'), array('..', '.')) as $item) {
					rename($nextcloud_tmp_dir.'/nextcloud/'.$item, './'.$item);
				}
				rmdir($nextcloud_tmp_dir.'/nextcloud');
			} else {
				rename($nextcloud_tmp_dir.'/nextcloud', './'.$directory);
			}
			// Delete the tmp folder
			rmdir($nextcloud_tmp_dir);
		} else {
			$error.='unzip of nextcloud source file failed.<br />';
		}

		// deleting zip file
		$result=@unlink('nc.zip');
		if($result==false) $error.='deleting of nc.zip failed.<br />';
		return($error);
	}


	/**
	* Downloads a file and stores it in the local filesystem
	* @param string $url
	* @param string$path
	* @return string with error messages
	*/
	static public function getFile($url,$path) {
		$error='';

		$fp = fopen ($path, 'w+');
		$ch = curl_init($url);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);
		curl_setopt($ch, CURLOPT_FILE, $fp);
		curl_setopt($ch, CURLOPT_USERAGENT, $_SERVER['HTTP_USER_AGENT']);
		if (Setup::isCertInfoAvailable()){
			curl_setopt($ch, CURLOPT_CERTINFO, TRUE);
		}
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, TRUE);
		$data=curl_exec($ch);
		$curlError=curl_error($ch);
		curl_close($ch);
		fclose($fp);

		if($data==false){
			$error.='download of Nextcloud source file failed.<br />'.$curlError;
		}
		return($error.$curlError);

	}


	/**
	* Shows the html header of the setup page
	*/
	static public function showHeader() {
		echo('
        <!DOCTYPE html>
        <html>
            <head>
                <title>Nextcloud Setup</title>
                <meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
                <link rel="icon" type="image/png" href="https://nextcloud.com/wp-content/themes/next/assets/img/common/favicon.png" />
                <meta http-equiv="X-UA-Compatible" content="IE=edge">
                <meta name="referrer" content="never">
                <meta name="theme-color" content="#0082c9">
				<style type="text/css">
				/* Copyright (c) 2011, Jan-Christoph Borchardt, http://jancborchardt.net
 This file is licensed under the Affero General Public License version 3 or later.
 See the COPYING-README file. */

html, body, div, span, object, iframe, h1, h2, h3, h4, h5, h6, p, blockquote, pre, a, abbr, acronym, address, code, del, dfn, em, img, q, dl, dt, dd, ol, ul, li, fieldset, form, label, legend, table, caption, tbody, tfoot, thead, tr, th, td, article, aside, dialog, figure, footer, header, hgroup, nav, section { margin:0; padding:0; border:0; outline:0; font-weight:inherit; font-size:100%; font-family:inherit; vertical-align:baseline; cursor:default; }
html, body { height:100%; }
article, aside, dialog, figure, footer, header, hgroup, nav, section { display:block; }
body { line-height:1.5; }
table { border-collapse:separate; border-spacing:0; white-space:nowrap; }
caption, th, td { text-align:left; font-weight:normal; }
table, td, th { vertical-align:middle; }
a { border:0; color:#000; text-decoration:none;}
a, a *, input, input *, select, .button span, label { cursor:pointer; }
ul { list-style:none; }

body {
	background-color: #ffffff;
	font-weight: 400;
	font-size: .8em;
	line-height: 1.6em;
	font-family: \'Open Sans\', Frutiger, Calibri, \'Myriad Pro\', Myriad, sans-serif;
	color: #000;
	height: auto;
}

#body-login {
	text-align: center;
	background-color: #0082c9;
	background-image: url(data:image/png;base64,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);
	background-position: 50% 50%;
	background-repeat: no-repeat;
	background-size: cover;
}

.float-spinner {
	height: 32px;
	display: none;
}
#body-login .float-spinner {
	margin-top: -32px;
	padding-top: 32px;
}


/* LOG IN & INSTALLATION ------------------------------------------------------------ */

/* Some whitespace to the top */
#body-login #header {
	padding-top: 100px;
}
#body-login {
	background-attachment: fixed; /* fix background gradient */
	height: 100%; /* fix sticky footer */
}

/* Dark subtle label text */
#body-login p.info,
#body-login form fieldset legend,
#body-login #datadirContent label,
#body-login form fieldset .warning-info,
#body-login form input[type="checkbox"]+label {
	text-align: center;
	color: #fff;
}
/* overrides another !important statement that sets this to unreadable black */
#body-login form .warning input[type="checkbox"]:hover+label,
#body-login form .warning input[type="checkbox"]:focus+label,
#body-login form .warning input[type="checkbox"]+label {
	color: #fff !important;
}

#body-login form {
    width: 280px;
    margin: 0 auto;
}
#body-login .infogroup {
	margin-bottom: 15px;
}

#body-login p#message img {
	vertical-align: middle;
	padding: 5px;
}

#body-login div.buttons {
	text-align: center;
}
#body-login p.info {
	width: 22em;
	margin: 0 auto;
	padding-top: 20px;
	-webkit-user-select: none;
	-moz-user-select: none;
	-ms-user-select: none;
	user-select: none;
}
#body-login p.info a {
	font-weight: 600;
	padding: 13px;
	margin: -13px;
}

/* position log in button as confirm icon in right of password field */
#body-login #submit.login {
	position: absolute;
	right: 0;
	top: 0;
	border: none;
	background-color: transparent;
	-ms-filter: "progid:DXImageTransform.Microsoft.Alpha(Opacity=30)";
	opacity: .3;
}
#body-login #submit.login:hover,
#body-login #submit.login:focus {
	-ms-filter: "progid:DXImageTransform.Microsoft.Alpha(Opacity=70)";
	opacity: .7;
}
#body-login input[type="password"] {
	padding-right: 40px;
	box-sizing: border-box;
	min-width: 269px;
}

#body-login form fieldset {
	margin-bottom: 20px;
	text-align: left;
	-webkit-user-select: none;
	-moz-user-select: none;
	-ms-user-select: none;
	user-select: none;
}
#body-login form #sqliteInformation {
	margin-top: -20px;
	margin-bottom: 20px;
}
#body-login form #adminaccount {
	margin-bottom: 15px;
}
#body-login form fieldset legend, #datadirContent label {
	width: 100%;
}
#body-login #datadirContent label {
	display: block;
	margin: 0;
}
#body-login form #datadirField legend {
	margin-bottom: 15px;
}
#body-login #showAdvanced {
	padding: 13px; /* increase clickable area of Advanced dropdown */
}
#body-login #showAdvanced img {
	vertical-align: bottom; /* adjust position of Advanced dropdown arrow */
	margin-left: -4px;
}
#body-login .icon-info-white {
	padding: 10px;
}

/* strengthify wrapper */
#body-login .strengthify-wrapper {
	display: inline-block;
	position: relative;
	left: 15px;
	top: -23px;
	width: 250px;
}

/* tipsy for the strengthify wrapper looks better with following font settings */
#body-login .tipsy-inner {
	font-weight: bold;
	color: #ccc;
}

/* General new input field look */
#body-login input[type="text"],
#body-login input[type="password"],
#body-login input[type="email"] {
	border: none;
	font-weight: 300;
}

/* keep the labels for screen readers but hide them since we use placeholders */
label.infield {
	display: none;
}

#body-login p.info a {
    color: #fff;
}

#body-login footer .info {
	white-space: nowrap;
}

/* Warnings and errors are the same */
#body-login .warning,
#body-login .update,
#body-login .error {
	display: block;
	padding: 10px;
	background-color: rgba(0,0,0,.3);
	color: #fff;
	text-align: left;
	border-radius: 3px;
	cursor: default;
}

#body-login .v-align {
	width: inherit;
}

.warning legend,
.warning a,
.error a {
	color: #fff !important;
	font-weight: 600 !important;
}

.warning-input {
	border-color: #ce3702 !important;
}

/* Fixes for log in page, TODO should be removed some time */
#body-login .warning {
	margin: 0 7px 5px 4px;
}
#body-login .warning legend {
	-ms-filter: "progid:DXImageTransform.Microsoft.Alpha(Opacity=100)";
	opacity: 1;
}
#body-login a.warning {
	cursor: pointer;
}

/* Log in and install button */
#body-login input {
	font-size: 20px;
	margin: 5px;
	padding: 11px 10px 9px;
}
#body-login input[type="text"],
#body-login input[type="password"] {
	width: 249px;
}
#body-login input.login {
	width: auto;
	float: right;
}
#body-login input[type="submit"] {
	padding: 10px 20px; /* larger log in and installation buttons */
}

/* Sticky footer */
#body-login .wrapper {
	min-height: 100%;
	margin: 0 auto -70px;
	width: 300px;
}
#body-login footer, #body-login .push {
	height: 70px;
}


/* INPUTS */

/* specifically override browser styles */
input, textarea, select, button {
	font-family: \'Open Sans\', Frutiger, Calibri, \'Myriad Pro\', Myriad, sans-serif;
}

input[type="text"],
textarea,
select,
button, .button,
input[type="submit"],
input[type="button"],
.pager li a {
	width: 130px;
	margin: 3px 3px 3px 0;
	padding: 7px 6px 5px;
	font-size: 13px;
	background-color: #fff;
	color: #333;
	border: 1px solid #ddd;
	outline: none;
	border-radius: 3px;
}
input[type="hidden"] {
	height: 0;
	width: 0;
}
input[type="text"] {
	background: #fff;
	color: #555;
	cursor: text;
	font-family: inherit; /* use default ownCloud font instead of default textarea monospace */
}
input[type="text"] {
	-webkit-appearance:textfield;
	-moz-appearance:textfield;
	box-sizing:content-box;
}
input[type="text"]:hover, input[type="text"]:focus, input[type="text"]:active {
	color: #333;
	-ms-filter: "progid:DXImageTransform.Microsoft.Alpha(Opacity=100)";
	opacity: 1;
}

input[type="checkbox"].checkbox {
	position: absolute;
	left:-10000px;
	top: auto;
	width: 1px;
	height: 1px;
	overflow: hidden;
}

/* BUTTONS */
input[type="submit"], input[type="button"],
button, .button,
#quota, select, .pager li a {
	width: auto;
	min-width: 25px;
	padding: 5px;
	background-color: rgba(240,240,240,.9);
	font-weight: 600;
	color: #555;
	border: 1px solid rgba(240,240,240,.9);
	cursor: pointer;
}
select, .button.multiselect {
	font-weight: 400;
}
input[type="submit"]:hover, input[type="submit"]:focus,
input[type="button"]:hover, input[type="button"]:focus,
.button:hover, .button:focus,
.button a:focus,
select:hover, select:focus, select:active {
	background-color: rgba(255, 255, 255, .95);
	color: #111;
}
input[type="submit"] img, input[type="button"] img, button img, .button img { cursor:pointer; }
#header .button {
	border: none;
	box-shadow: none;
}

/* Primary action button, use sparingly */
.primary, input[type="submit"].primary, input[type="button"].primary, button.primary, .button.primary {
	border: 1px solid #fff;
	background-color: #00a2e9;
	color: #ddd;
}
.primary:hover, input[type="submit"].primary:hover, input[type="button"].primary:hover, button.primary:hover, .button.primary:hover,
.primary:focus, input[type="submit"].primary:focus, input[type="button"].primary:focus, button.primary:focus, .button.primary:focus {
	background-color: #0092d9;
	color: #fff;
}

                </style>
			</head>

			<body id="body-login">
		');
	}


	/**
	* Shows the html footer of the setup page
	*/
	static public function showFooter() {
		echo('
		<footer><p class="info"><a href="https://nextcloud.com/" target="_blank" rel="noreferrer">Nextcloud</a> &ndash; a safe home for all your data</p></footer>
		</body>
		</html>
		');
	}


	/**
	* Shows the html content part of the setup page
	* @param string $title
	* @param string $content
	* @param string $nextpage
	*/
	static public function showContent($title, $content, $nextpage=''){
		echo('
		<script>
			var validateForm = function(){
				if (typeof urlNotExists === "undefined"){
					return true;
				}
				urlNotExists(
					window.location.href,
					function(){
						window.location.assign(document.forms["install"]["directory"].value);
					}
				);
				return false;
			}
		</script>
		<div id="login" class="wrapper">
		<div class="v-align">
			<header><div id="header">
<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<svg xmlns="http://www.w3.org/2000/svg" version="1.1" xml:space="preserve" height="100" width="210" enable-background="new 0 0 196.6 72" y="0px" x="0px" viewBox="0 0 62.000002 34"><path style="color-rendering:auto;text-decoration-color:#000000;color:#000000;isolation:auto;mix-blend-mode:normal;shape-rendering:auto;solid-color:#000000;block-progression:tb;text-decoration-line:none;image-rendering:auto;white-space:normal;text-indent:0;enable-background:accumulate;text-transform:none;text-decoration-style:solid" fill="#fff" d="m31.6 4.0001c-5.95 0.0006-10.947 4.0745-12.473 9.5549-1.333-2.931-4.266-5.0088-7.674-5.0092-4.6384 0.0005-8.4524 3.8142-8.453 8.4532-0.0008321 4.6397 3.8137 8.4544 8.4534 8.455 3.4081-0.000409 6.3392-2.0792 7.6716-5.011 1.5261 5.4817 6.5242 9.5569 12.475 9.5569 5.918 0.000457 10.89-4.0302 12.448-9.4649 1.3541 2.8776 4.242 4.9184 7.6106 4.9188 4.6406 0.000828 8.4558-3.8144 8.4551-8.455-0.000457-4.6397-3.8154-8.454-8.4551-8.4533-3.3687 0.0008566-6.2587 2.0412-7.6123 4.9188-1.559-5.4338-6.528-9.4644-12.446-9.464zm0 4.9623c4.4687-0.000297 8.0384 3.5683 8.0389 8.0371 0.000228 4.4693-3.5696 8.0391-8.0389 8.0388-4.4687-0.000438-8.0375-3.5701-8.0372-8.0388 0.000457-4.4682 3.5689-8.0366 8.0372-8.0371zm-20.147 4.5456c1.9576 0.000226 3.4908 1.5334 3.4911 3.491 0.000343 1.958-1.533 3.4925-3.4911 3.4927-1.958-0.000228-3.4913-1.5347-3.4911-3.4927 0.0002284-1.9575 1.5334-3.4907 3.4911-3.491zm40.205 0c1.9579-0.000343 3.4925 1.533 3.4927 3.491 0.000457 1.9584-1.5343 3.493-3.4927 3.4927-1.958-0.000228-3.4914-1.5347-3.4911-3.4927 0.000221-1.9575 1.5335-3.4907 3.4911-3.491z"/></svg>

			</div></header><br />
			<p style="text-align:center; font-size:28px; color:#fff; font-weight:bold;">'.$title.'</p><br />
			<form method="get" name="install" onsubmit="return validateForm();">
			    <p class="warning" style="text-align:center; font-size:13px;">'.$content.'</p>
				<input type="hidden" name="step" value="'.$nextpage.'" />
		');

		if($nextpage === 2) {
			echo ('<p class="warning">Enter a single "." to install in the current directory, or enter a subdirectory to install to:</p>
				<input type="text" name="directory" value="nextcloud" required="required" />');
		}
		if($nextpage === 3) {
			echo ('<input type="hidden" value="'.$_GET['directory'].'" name="directory" />');
		}

		if($nextpage<>'') echo('<input type="submit" id="submit" class="primary button" value="Next" />');

		echo('
		</form>
		<div class="push"></div>
		</div>
		</div>
		');
	}

	/**
	 * JS function to check if user deleted this script
	 * N.B. We can't reload the page to check this with PHP:
	 * once script is deleted we end up with 404
	 */
	static public function showJsValidation(){
		echo '
		<script>
			var urlNotExists = function(url, callback){
				var xhr = new XMLHttpRequest();
				xhr.open(\'HEAD\', encodeURI(url));
				xhr.onload = function() {
					if (xhr.status === 404){
						callback();
					}
				};
				xhr.send();
			};
		</script>
		';
	}


	/**
	* Shows the welcome screen of the setup wizard
	*/
	static public function showWelcome() {
		$txt='Welcome to the Setup Wizard for<br /><b>Nextcloud '.NC_VERSION.'</b>!<br /><br />This wizard will:<br />1. Check the server dependencies<br />2. Download Nextcloud<br />3. Install Nextcloud in a few simple steps';
		Setup::showContent('Setup Wizard',$txt,1);
	}


	/**
	* Shows the check dependencies screen
	*/
	static public function showCheckDependencies() {
		$error=Setup::checkDependencies();
		if($error=='') {
			$txt='All Nextcloud dependencies found';
			Setup::showContent('Dependency check',$txt,2);
		}else{
			$txt='Dependencies not found.<br />'.$error;
			Setup::showContent('Dependency check',$txt);
		}
	}


	/**
	* Shows the install screen
	*/
	static public function showInstall() {
		$error=Setup::install();

		if($error=='') {
			$txt='Nextcloud is now installed';
			Setup::showContent('Success',$txt,3);
		}else{
			$txt='Nextcloud is NOT installed<br />'.$error;
			Setup::showContent('Error',$txt);
		}
	}

	/**
	 * Shows the redirect screen
	 */
	static public function showRedirect() {
		// delete own file
		@unlink(__FILE__);
		clearstatcache();
		if (file_exists(__FILE__)){
			Setup::showJsValidation();
			Setup::showContent(
				'Warning',
				'Failed to remove installer script. Please remove ' . __FILE__ . ' manually',
				3
			);
		} else {
			// redirect to Nextcloud
			header("Location: " . $_GET['directory']);
		}
	}

}


// read the step get variable
$step = isset($_GET['step']) ? $_GET['step'] : 0;

// show the header
Setup::showHeader();

// show the right step
if     ($step==0) Setup::showWelcome();
elseif ($step==1) Setup::showCheckDependencies();
elseif ($step==2) Setup::showInstall();
elseif ($step==3) Setup::showRedirect();
else  echo('Internal error. Please try again.');

// show the footer
Setup::showFooter();
